module bounce


//	**************************************************************************************************
//
//	A program that creates two interactive processes that bounce balls in an open-ended barrel.
//
//	The program has been written in Clean 1.3.1 and uses the Clean Standard Object I/O library 1.0.1
//	
//	**************************************************************************************************


import StdEnv, StdIO
import bounceDraw


::	Message									// The message type:
	=	BallsArrive [Ball]					// balls that have crossed process border
	|	BounceOpened						// the other bounce process has been created
	|	QuitBounce							// quit the bounce process
::	Local									// The local program state:
	=	{	talkTo	:: !RId Message			// who to play with
		,	barrel	:: !Barrel				// the shape of the barrel
		,	balls	:: ![Ball]				// the balls in the barrel
		}
::	NoState									// NoState is a simple singleton type constructor
	=	NoState
::	*Bounce	x
	:==	PSt Local x							// Synonym for PSt


//	Create the initial interactive process:

Start :: *World -> *World
Start world
	#	(rIdA,world)	= openRId world
	#	(rIdB,world)	= openRId world
	#	(wId, world)	= openId  world
	#	(tId, world)	= openId  world
	=	startProcesses	[	bounce wId tId rIdB rIdA "Bounce B" RightTop rightBarrelSetUp
						,	bounce wId tId rIdA rIdB "Bounce A" LeftTop  leftBarrelSetUp
						]	world

bounce :: Id Id (RId Message) (RId Message) Title ItemLoc (Barrel,[Ball])
		-> ProcessGroup (SDIProcess (Window NilLS))
bounce wId tId me you name itemLoc (barrel,balls)
	=	ProcessGroup 0 (SDIProcess initLocal NoState window [initIO] [ProcessHelp bounceHelp])
where
	barrelDomain	= barrel.bDomain
	barrelSize		= rectangleSize barrelDomain
	maxSize			= maxFixedWindowSize
	windowSize		= {w=min barrelSize.w (maxSize.w/2),h=min barrelSize.h (maxSize.h/2)}
	splitWalls		= splitWallsInBarrel barrel
	initLocal		= {talkTo=you,barrel=barrel,balls=balls}
	
	initIO ps
		#	(error,ps)		= openMenu undef menu ps
		|	error<>NoError
			=	abort "bounce could not open menu."
		#	(error,ps)		= openTimer undef timer ps
		|	error<>NoError
			=	abort "bounce could not open timer."
		#	(error,ps)		= openReceiver undef receiver ps
		|	error<>NoError
			=	abort "bounce could not open receiver."
		|	otherwise
			=	ps
	
//	window defines the window that displays the barrel and the current balls.
	window	=	Window name NilLS
					[	WindowId		wId
					,	WindowLook		(updateBalls initLocal)
					,	WindowResize
					,	WindowSize		windowSize
					,	WindowPos		(itemLoc,zero)
					]
	
	updateBalls :: Local SelectState UpdateState -> [DrawFunction]
	updateBalls {balls,barrel} _ {oldFrame,newFrame,updArea}
		=	[drawBarrel area scale barrel : map (drawBall scale domain.corner1) balls]
	where
		domain		= barrel.bDomain
		windowSize	= rectangleSize newFrame
		barrelSize	= rectangleSize domain
		scale		= scaleSize windowSize barrelSize
		area		= if (oldFrame==newFrame) updArea [newFrame]
	
//	menu defines the menu. It contains only the quit command to terminate the application.
	menu	=	Menu name (MenuItem "Quit" [MenuShortKey 'q',MenuFunction (noLS quit)]) []
	where
		quit :: (Bounce .x) -> Bounce .x
		quit bounce=:{ls={talkTo}}
			=	closeProcess (snd (syncSend talkTo QuitBounce bounce))
	
//	timer defines the timer that will calculate the movements of the current balls as often as possible.
	timer	=	Timer 0 NilLS 
					[	TimerId			tId
					,	TimerFunction	(noLS1 (bounceBalls splitWalls))
					]
	where
		bounceBalls :: !(![SingleWall],![SingleWall]) NrOfIntervals (Bounce .x) -> Bounce .x
		bounceBalls splitWalls _ bounce=:{ls=local=:{talkTo,balls,barrel},io}
			#	(windowSize,io)	= getWindowViewSize wId io
				scale			= scaleSize windowSize barrelSize
				eraseOld		= map (eraseBall scale base) balls
				drawNew			= map (drawBall  scale base) ins
				local			= {local & balls=ins}
			#	io				= drawInWindow  wId (eraseOld++drawNew) io
			#	io				= setWindowLook wId False (updateBalls local) io
			#	bounce			= {bounce & ls=local,io=io}
			|	isEmpty outs
				=	bounce
			|	otherwise
				=	snd (syncSend talkTo (BallsArrive outs) bounce)
		where
			nextBallPos			= nextBallPositions splitWalls balls
			ballsMoved			= map moveBall nextBallPos
			domain				= barrel.bDomain
			base				= domain.corner1
			barrelSize			= rectangleSize domain
			(ins,outs)			= splitBallsInBarrel domain ballsMoved
	
//	receiver defines the receiver that will receive new balls and termination requests.
	receiver	=	Receiver me (noLS1 (receive splitWalls)) []
	where
		receive :: !(![SingleWall],![SingleWall]) !Message !(Bounce .x) -> Bounce .x
		receive (horizontal,vertical) (BallsArrive newBalls) bounce=:{ls}
			#!	newBalls = map correctBall newBalls
			=	{bounce & ls={ls & balls=newBalls++ls.balls}}
		where
			correctBall :: !Ball -> Ball
			correctBall ball=:{bCenter,bSpeed}
				#	ball = {ball & bCenter=movePoint (~bSpeed) bCenter}
				#	ball = checkVerticalWalls   vertical   ball
				#	ball = checkHorizontalWalls horizontal ball
				#	ball = moveBall ball
				=	ball
		receive _ BounceOpened bounce
			=	appPIO (enableTimer tId) bounce
		receive _ QuitBounce bounce
			=	closeProcess bounce
	
//	bounceHelp opens a dialog that tells something about this application.
	bounceHelp :: (Bounce .x) -> Bounce .x
	bounceHelp bounce
		#	(okId, bounce)	= accPIO openId bounce
		#	(error,bounce)	= openModalDialog undef (dDef okId) bounce
		|	error<>NoError
			=	abort "bounce could not open About bounce dialog."
		|	otherwise
			=	bounce
	where
		dDef okId
			=	Dialog "About bounce"
					(	TextControl   "This is a Clean program"
										   [ControlPos center]
					:+:	ButtonControl "Ok" [ControlId okId,ControlPos center,ControlFunction (noLS close)]
					)
					[	WindowOk okId
					]
		center	= (Center,zero)
		
		close :: (Bounce .x) -> Bounce .x
		close bounce
			#	(Just id,bounce)= accPIO getActiveWindow  bounce
			#	bounce			= closeWindow id bounce
			=	bounce


//	Determine which balls are inside and which are outside the barrel:

splitBallsInBarrel :: !ViewDomain ![Ball] -> (![Ball],![Ball])
splitBallsInBarrel domain balls
	=	seq (map (ballInOrOut domain) balls) ([],[])
where
	ballInOrOut :: !ViewDomain !Ball !(![Ball],![Ball]) -> (![Ball],![Ball])
	ballInOrOut {corner1={x=left,y=top},corner2={x=right,y=bottom}} ball=:{bCenter} (ins,outs)
		|	between bCenter.x left right && between bCenter.y top bottom
			=	([ball:ins],outs)
		|	otherwise
			=	(ins,[ball:outs])

nextBallPositions :: !(![SingleWall],![SingleWall]) ![Ball] -> [Ball]
nextBallPositions (horizontal,vertical) balls
	=	map (checkHorizontalWalls horizontal) (
		map (checkVerticalWalls   vertical)   (
		computeNextBallPositions [] balls))
where
	computeNextBallPositions :: ![Ball] ![Ball] -> [Ball]
	computeNextBallPositions ballsDone [ball:balls]
		=	computeNextBallPositions [ballDone:newBallsDone] newBalls
	where
		(newBallsDone,newBalls,ballDone) = checkBallCollisions ballsDone balls ball
		
		checkBallCollisions :: ![Ball] ![Ball] !Ball -> (![Ball],![Ball],!Ball)
		checkBallCollisions balls1 balls2 ball
			=	(newBalls1,newBalls2,ball2)
		where
			(newBalls1,ball1)	= checkBallCollision balls1 ball
			(newBalls2,ball2)	= checkBallCollision balls2 ball1
			
			checkBallCollision :: ![Ball] !Ball -> (![Ball],!Ball)
			checkBallCollision [ball2=:{bCenter=center2,bRadius=radius2,bSpeed=step2}:others]
								ball1=:{bCenter=center1,bRadius=radius1,bSpeed=step1}
				|	dist (moveBall ball1).bCenter center2<=toReal (radius1+radius2)
					#	(others,ball1)	= checkBallCollision others {ball1 & bSpeed=step2}
					=	([{ball2 & bSpeed=step1}:others],ball1)
				|	otherwise
					#	(others,ball1)	= checkBallCollision others ball1
					=	([ball2:others],ball1)
			checkBallCollision others ball
				=	(others,ball)
	computeNextBallPositions ballsDone _
		=	ballsDone

checkHorizontalWalls :: ![SingleWall] !Ball -> Ball
checkHorizontalWalls [((a,b),interior):walls] ball=:{bCenter,bRadius,bSpeed}
	|	interior<>startInterior
		=	checkHorizontalWalls walls ball
	|	not collision
		=	checkHorizontalWalls walls ball1
	|	otherwise
		=	ball1
where
	c				= (moveBall ball).bCenter
	speed1			= if collision {bSpeed & vy=0-bSpeed.vy} bSpeed
	collision		= (between c.x (a.x-bRadius) (b.x+bRadius))
					  &&	(sign bSpeed.vy<>interior)
					  		&&	(if posSign (a.y+signRadius>=c.y)
					  						(a.y+signRadius<=c.y))
	signRadius		= interior*bRadius
	posSign			= interior>0
	startInterior	= sign (bCenter.y-a.y)
	ball1			= {ball & bSpeed=speed1}
checkHorizontalWalls _ ball
	=	ball

checkVerticalWalls :: ![SingleWall] !Ball -> Ball
checkVerticalWalls [((a,b),interior):walls] ball=:{bCenter,bRadius,bSpeed}
	|	interior<>startInterior
		=	checkVerticalWalls walls ball
	|	not collision
		=	checkVerticalWalls walls ball1
	|	otherwise
		=	ball1
where
	c				= (moveBall ball).bCenter
	speed1			= if collision {bSpeed & vx=0-bSpeed.vx} bSpeed
	collision		= (between c.y (a.y-bRadius) (b.y+bRadius))
					  &&	((sign bSpeed.vx<>interior)
							&&	(if posSign (a.x+signRadius>=c.x)
											(a.x+signRadius<=c.x)))
	signRadius		= interior*bRadius
	posSign			= interior>0
	startInterior	= sign (bCenter.x-a.x)
	ball1			= {ball & bSpeed=speed1}
checkVerticalWalls _ ball
	=	ball

moveBall :: !Ball -> Ball
moveBall ball=:{bCenter,bSpeed}
	=	{ball & bCenter=movePoint bSpeed bCenter}
